function [xECI] = Syn2ECI(t, xSyn, mu)
%
% Transformation from PBR4BP Synodic r.f. to ECI r.f.
%
%DESCRIPTION:
%This code provides transformation from PBR4BP Synodic r.f. to Earth
%Centred Inertial (ECI) r.f.
%
%PROTOTYPE
%   [xECI] = Syn2ECI(t, xSyn, mu)
%
%--------------------------------------------------------------------------
% INPUTS:
%   t          [1xn]       Time Instant              [-]
%   xSyn       [nx4]       State Vector (Synodic)    [-]
%   mu         [1x1]       System Mass Parameter     [-]
%--------------------------------------------------------------------------
% OUTPUTS:
%   xECI       [nx4]       State Vector (ECI)        [-]
%--------------------------------------------------------------------------
%
%NOTES:
% (none)
%
%CALLED FUNCTIONS:
% (none)
%
%UPDATES:
% (none)
%
%REFERENCES:
% [1] "On Optimal Two-Impulse Earth-Moon Transfers in a Four-Body Model",
%     Francesco Topputo.
%
%AUTHOR(s):
%Luigi De Maria, 2022
%

%% Transformation

%Memory Allocation
[r,c] = size(xSyn);                       %Input Vector Sizes
xECI  = zeros(r,c);

%State Transformation
xECI(:,1) = (xSyn(:,1) + mu).*cos(t(:)) - xSyn(:,2).*sin(t(:));
xECI(:,2) = (xSyn(:,1) + mu).*sin(t(:)) + xSyn(:,2).*cos(t(:));
xECI(:,3) = (xSyn(:,3) - xSyn(:,2)).*cos(t(:)) - (xSyn(:,4) + xSyn(:,1) + mu).*sin(t(:));
xECI(:,4) = (xSyn(:,3) - xSyn(:,2)).*sin(t(:)) + (xSyn(:,4) + xSyn(:,1) + mu).*cos(t(:));

end